package com.live.common.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.constant.ConstantValue;
import com.live.common.domain.ResponseData;
import com.live.common.domain.ResultPage;
import com.live.common.domain.dto.api.*;
import com.live.common.domain.entity.*;
import com.live.common.domain.request.CommonPage;
import com.live.common.domain.request.CommonStringId;
import com.live.common.enums.SystemConfigEnum;
import com.live.common.mapper.*;
import com.live.common.service.LeaguePointsRankService;
import com.live.common.utils.DateUtil;
import com.live.common.utils.match.CommonUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

@Slf4j
@Service
public class LeaguePointsRankServiceImpl implements LeaguePointsRankService {

    @Resource
    private UserMapper userMapper;
    @Resource
    private TeamMapper teamMapper;
    @Resource
    private MatchMapper matchMapper;
    @Resource
    private FollowMapper followMapper;
    @Resource
    private SportsMapper sportsMapper;
    @Resource
    private QuizRecordMapper quizRecordMapper;
    @Resource
    private SystemConfigMapper systemConfigMapper;
    @Resource
    private FootballLeagueRankMapper footballLeagueRankMapper;
    @Resource
    private BasketballLeagueRankMapper basketballLeagueRankMapper;

    private final Integer SOURCE_TYPE_AL_STAT = 1;

    @Override
    public ResponseData<?> getLeagueType(HttpServletRequest request) {
        List<SystemConfig> systemConfigs = systemConfigMapper.selectList(Wrappers.<SystemConfig>lambdaQuery()
                .eq(SystemConfig::getConfigName, SystemConfigEnum.FOOTBALL_LEAGUE_RANK_COMPETITION.getTitle())
                .eq(SystemConfig::getDeleted, 0)
        );

        List<SystemConfig> basketballConfigs = systemConfigMapper.selectList(Wrappers.<SystemConfig>lambdaQuery()
                .eq(SystemConfig::getConfigName, SystemConfigEnum.BASKETBALL_LEAGUE_RANK_COMPETITION.getTitle())
                .eq(SystemConfig::getDeleted, 0)
        );
        if (systemConfigs.size() + basketballConfigs.size() == 0)
            return ResponseData.successResponse();

        List<GetLeagueTypeResponse> responses = new ArrayList<>();

        List<Sports> sports = sportsMapper.selectList(Wrappers.<Sports>lambdaQuery()
                .eq(Sports::getDeleted, 0)
                .eq(Sports::getSourceType, SOURCE_TYPE_AL_STAT)
                .in(Sports::getSourceId, systemConfigs.stream().map(SystemConfig::getConfigValue).collect(Collectors.toList()))
                .eq(Sports::getSportsId, 0)
        );

        responses = sports.stream().map(b -> GetLeagueTypeResponse.builder()
                .id(b.getId())
                .name(b.getCompetitionShortName())
                .sportsType(b.getSportsId())
                .build()).collect(Collectors.toList());

        List<Sports> sportBasketball = sportsMapper.selectList(Wrappers.<Sports>lambdaQuery()
                .eq(Sports::getDeleted, 0)
                .eq(Sports::getSourceType, SOURCE_TYPE_AL_STAT)
                .in(Sports::getSourceId, basketballConfigs.stream().map(SystemConfig::getConfigValue).collect(Collectors.toList()))
                .eq(Sports::getSportsId, 1)
        );
        if (sportBasketball.size() > 0) {
            responses.addAll(sportBasketball.stream().map(b -> GetLeagueTypeResponse.builder()
                    .id(b.getId())
                    .name(b.getCompetitionShortName())
                    .sportsType(b.getSportsId())
                    .build()).collect(Collectors.toList()));
        }
        return ResponseData.successResponse(responses);
    }

    @Override
    public ResponseData<?> getLeaguePointsRank(CommonStringId commonStringId, HttpServletRequest request) {
        Sports sports = sportsMapper.selectById(commonStringId.getId());
        if (sports == null)
            return ResponseData.fail400Response("ID错误");

        List<FootballLeagueRank> footballLeagueRanks = footballLeagueRankMapper.selectList(Wrappers.<FootballLeagueRank>lambdaQuery()
                .eq(FootballLeagueRank::getSportsId, sports.getId())
                .eq(FootballLeagueRank::getDeleted, 0)
                .orderByDesc(FootballLeagueRank::getPoints)
        );

        List<LeaguePointsRankFootballResponse> responses = new LinkedList<>();

        for (char letter = 'A'; letter <= 'Z'; letter++) {
            String key = Character.toString(letter);

            List<LeaguePointsRankFootballResponse.RankFootballResponse> rankFootball = new LinkedList<>();
            for (FootballLeagueRank footballLeagueRank : footballLeagueRanks) {
                if (footballLeagueRank.getGroupName().trim().equals(key)) {
                    Team team = teamMapper.selectById(footballLeagueRank.getTeamId());
                    if (team == null) {
                        log.error("getLeaguePointsRank ERROR:{} is null", footballLeagueRank.getTeamId());
                        continue;
                    }

                    rankFootball.add(LeaguePointsRankFootballResponse.RankFootballResponse.builder()
                            .draws(footballLeagueRank.getDraws())
                            .losts(footballLeagueRank.getLosts())
                            .points(footballLeagueRank.getPoints())
                            .teamName(team.getTeamName())
                            .teamIcon(team.getTeamIcon())
                            .wins(footballLeagueRank.getWins())
                            .build());
                }
            }

            if (rankFootball.size() > 0) {
                responses.add(LeaguePointsRankFootballResponse.builder()
                        .groupName(key)
                        .rankFootball(rankFootball)
                        .build());
            }
        }
        return ResponseData.successResponse(responses);
    }

    @Override
    public ResponseData<?> getLeaguePointsRankBasketball(CommonStringId commonStringId, HttpServletRequest request) {
        Sports sports = sportsMapper.selectById(commonStringId.getId());
        if (sports == null)
            return ResponseData.fail400Response("ID错误");

        List<BasketballLeagueRank> rankList = basketballLeagueRankMapper.selectList(Wrappers.<BasketballLeagueRank>lambdaQuery()
                .eq(BasketballLeagueRank::getDeleted, 0)
                .eq(BasketballLeagueRank::getSportsId, sports.getId())
                .orderByAsc(BasketballLeagueRank::getPosition)
        );

        Map<String, List<BasketballLeagueRank>> stageNames = new HashMap<>();
        for (BasketballLeagueRank rank : rankList) {
            stageNames.computeIfAbsent(rank.getStageName(), b -> new ArrayList<>()).add(rank);
        }

        List<LeaguePointsRankBasketballResponse.StageBasketball> stageBasketballs = new LinkedList<>();
        for(String stage : stageNames.keySet()){
            List<LeaguePointsRankBasketballResponse.GroupBasketball> basketballs = new LinkedList<>();
            Map<String, List<LeaguePointsRankBasketballResponse.RankBasketballResponse>> groupMap = new HashMap<>();

            for (BasketballLeagueRank rank : stageNames.get(stage)) {
                try {
                    Team team = teamMapper.selectById(rank.getTeamId());
                    groupMap.computeIfAbsent(rank.getGroupName(), b -> new ArrayList<>()).add(LeaguePointsRankBasketballResponse.RankBasketballResponse
                            .builder()
                            .losts(rank.getLosts())
                            .points(rank.getPosition())
                            .pointsFor(rank.getPointsFor())
                            .pointsDiff(rank.getPointsDiff())
                            .teamIcon(team.getTeamIcon())
                            .teamName(team.getTeamName())
                            .wins(rank.getWins())
                            .build());
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }

            for (char letter = 'A'; letter <= 'Z'; letter++) {
                String key = letter + "组";
                if(!groupMap.containsKey(key))
                    continue;

                basketballs.add(LeaguePointsRankBasketballResponse.GroupBasketball.builder()
                        .groupName(key)
                        .rankInfos(groupMap.get(key))
                        .build());
            }
            stageBasketballs.add(LeaguePointsRankBasketballResponse.StageBasketball.builder()
                    .stageName(stage)
                    .groups(basketballs)
                    .build());
        }

        return ResponseData.successResponse(LeaguePointsRankBasketballResponse.builder()
                .stages(stageBasketballs)
                .build());
    }

    @Override
    public ResponseData<?> getHotExpert(HttpServletRequest request) {
        List<QuizRecord> quizRecords = quizRecordMapper.selectList(Wrappers.<QuizRecord>lambdaQuery()
                .eq(QuizRecord::getDeleted, 0)
                .ge(QuizRecord::getCreateTime, DateUtil.getAddDayStartTime(-7))
        );

        List<GetHotExpertResponse> responses = new ArrayList<>();
        Map<String, AtomicInteger> userWins = new HashMap<>();
        Map<String, AtomicInteger> userSum = new HashMap<>();
        for (QuizRecord quiz : quizRecords) {
            Match match = matchMapper.selectById(quiz.getMatchId());
            if(match == null)
                continue;

            userWins.computeIfAbsent(quiz.getUserId(), b -> new AtomicInteger())
                    .addAndGet(CommonUtil.winOrLose(match, quiz.getBetPosition()) ? 1 : 0);
            userSum.computeIfAbsent(quiz.getUserId(), b -> new AtomicInteger()).addAndGet(1);
        }

        for (String userId : userWins.keySet()) {
            User user = userMapper.selectById(userId);
            long count = followMapper.selectCount(Wrappers.<Follow>lambdaQuery()
                    .eq(Follow::getDeleted, 0)
                    .eq(Follow::getFollowId, user.getId())
            );

            responses.add(GetHotExpertResponse.builder()
                    .anchorDesc(user.getUserDesc())
                    .anchorName(user.getUserName())
                    .anchorIcon(ConstantValue.completeAddress(user.getUserIcon()))
                    .anchorId(user.getId())
                    .follows((int) count)
                    .wins(userWins.get(userId).intValue())
                    .sum(userSum.get(userId).intValue())
                    .build());
        }
        return ResponseData.successResponse(responses);
    }

    @Override
    public ResponseData<?> getAnchorRecommend(CommonPage commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        long count = userMapper.selectCount(Wrappers.<User>lambdaQuery()
                .eq(User::getDeleted, 0)
                .eq(User::getType, 1)
                .eq(User::getHotAnchor, 1)
        );

        List<User> users = userMapper.selectList(Wrappers.<User>lambdaQuery()
                .eq(User::getDeleted, 0)
                .eq(User::getType, 1)
                .eq(User::getHotAnchor, 1)
                .orderByDesc(User::getHotNum)
                .last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()))
        );

        List<GetAnchorRecommendResponse> responses = users.stream().map(b -> GetAnchorRecommendResponse.builder()
                .anchorDesc(b.getUserDesc())
                .anchorIcon(ConstantValue.completeAddress(b.getUserIcon()))
                .anchorId(b.getId())
                .anchorName(b.getUserName())
                .hotNum(b.getHotNum())
                .build())
                .collect(Collectors.toList());
        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, responses));
    }

}
