package com.live.common.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.constant.ConstantValue;
import com.live.common.domain.ResponseData;
import com.live.common.domain.ResultPage;
import com.live.common.domain.dto.back.GetGiftRecordResponse;
import com.live.common.domain.dto.back.GiftResponse;
import com.live.common.domain.entity.Category;
import com.live.common.domain.entity.Gift;
import com.live.common.domain.entity.GiftRecord;
import com.live.common.domain.entity.User;
import com.live.common.domain.request.CommonPage;
import com.live.common.domain.request.back.AddGiftRequest;
import com.live.common.domain.request.back.GiftRecordRequest;
import com.live.common.mapper.GiftMapper;
import com.live.common.mapper.GiftRecordMapper;
import com.live.common.mapper.UserMapper;
import com.live.common.service.GiftService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.sql.Wrapper;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service
public class GiftServiceImpl implements GiftService {

    @Resource
    private UserMapper userMapper;
    @Resource
    private GiftMapper giftMapper;
    @Resource
    private GiftRecordMapper giftRecordMapper;

    @Override
    public ResponseData<?> getGiftList(CommonPage commonPage) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        long count = giftMapper.selectCount(Wrappers.<Gift>lambdaQuery()
                .eq(Gift::getDeleted, 0)
        );

        List<Gift> gifts = giftMapper.selectList(Wrappers.<Gift>lambdaQuery()
                .eq(Gift::getDeleted, 0)
                .orderByAsc(Gift::getGiftGold)
                .last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()))
        );

        List<GiftResponse> giftResponses = gifts.stream().map(gift -> {
            return GiftResponse.builder()
                    .giftGold(gift.getGiftGold())
                    .giftIcon(ConstantValue.completeAddress(gift.getGiftIcon()))
                    .giftName(gift.getGiftName())
                    .id(gift.getId())
                    .isOnline(gift.getIsOnline())
                    .deleted(gift.getDeleted())
                    .build();
        }).collect(Collectors.toList());

        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, giftResponses));
    }

    @Override
    public ResponseData<?> addGift(AddGiftRequest addGiftRequest) {
        giftMapper.insert(Gift.builder()
                .giftGold(addGiftRequest.getGiftGold())
                .giftIcon(addGiftRequest.getGiftIcon())
                .giftName(addGiftRequest.getGiftName())
                .isOnline(addGiftRequest.getIsOnline())
                .build());
        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> editGift(AddGiftRequest addGiftRequest) {
        Gift gift = giftMapper.selectById(addGiftRequest.getId());
        if (gift == null) {
            return ResponseData.fail400Response("ID 错误，未找到");
        }

        gift.setGiftGold(addGiftRequest.getGiftGold());
        gift.setGiftIcon(addGiftRequest.getGiftIcon());
        gift.setGiftName(addGiftRequest.getGiftName());
        gift.setIsOnline(addGiftRequest.getIsOnline());
        gift.setDeleted(addGiftRequest.getDeleted());
        giftMapper.updateById(gift);

        return ResponseData.successResponse();
    }

    @Override
    public ResponseData<?> getGiftRecord(GiftRecordRequest commonPage, HttpServletRequest request) {
        int jumpNum = (commonPage.getPageNum() - 1) * commonPage.getPageSize();

        QueryWrapper<GiftRecord> wrapper = new QueryWrapper<>();
        if(commonPage.getGiftId() != null && commonPage.getGiftId() != 0) {
            wrapper.eq("gift_id", commonPage.getGiftId());
        }
        if(commonPage.getAnchorId() != null && commonPage.getAnchorId() != 0) {
            wrapper.eq("host_user_id", commonPage.getAnchorId());
        }
        if(commonPage.getUserId() != null && commonPage.getUserId() != 0) {
            wrapper.eq("user_id", commonPage.getUserId());
        }
        wrapper.eq("deleted", 0);

        long count = giftRecordMapper.selectCount(wrapper);

        wrapper.last(String.format(" limit %s,%s", jumpNum, commonPage.getPageSize()));
        List<GiftRecord> giftRecords = giftRecordMapper.selectList(wrapper);

        List<GetGiftRecordResponse> getGiftRecord = giftRecords.stream().map(giftRecord -> {
            Gift gift = giftMapper.selectById(giftRecord.getGiftId());
            User anchor = userMapper.selectById(giftRecord.getHostUserId());
            User user = userMapper.selectById(giftRecord.getUserId());
            return GetGiftRecordResponse.builder()
                    .giftGold(giftRecord.getGiftGold())
                    .giftId(giftRecord.getGiftId())
                    .giftImg(ConstantValue.completeAddress(gift.getGiftIcon()))
                    .giftName(gift.getGiftName())

                    .anchorId(anchor.getId())
                    .anchorName(anchor.getUserName())

                    .userId(user.getId())
                    .userName(user.getUserName())
                    .build();
        }).collect(Collectors.toList());

        return ResponseData.successResponse(new ResultPage<>(commonPage.getPageNum(), commonPage.getPageSize(), (int) count, getGiftRecord));
    }


}
