package com.live.job.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.live.common.domain.entity.*;
import com.live.common.enums.SystemConfigEnum;
import com.live.common.mapper.*;
import com.live.common.service.RedisUtilsService;
import com.live.common.service.RoomService;
import com.live.common.utils.DateUtil;
import com.live.common.utils.IdGen;
import com.live.common.utils.RedisCommonChar;
import com.live.job.constant.CrawlPlatformEnum;
import com.live.job.entity.dto.SportLive360ListDto;
import com.live.job.service.CrawlMatchService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ThreadPoolExecutor;

import static org.springframework.http.MediaType.APPLICATION_JSON;

@Slf4j
@Service
public class CrawlSportLive360ServiceImpl implements CrawlMatchService {

    @Value("${sportlive360.clientID}")
    private String clientID;
    @Resource
    private RoomMapper roomMapper;
    @Resource
    private UserMapper userMapper;
    @Resource
    private TeamMapper teamMapper;
    @Resource
    private MatchMapper matchMapper;
    @Resource
    private RoomService roomService;
    @Resource
    private RestTemplate restTemplate;
    @Resource
    private RedisUtilsService redisUtilsService;
    @Resource
    private SystemConfigMapper systemConfigMapper;
    @Resource
    private ThreadPoolExecutor threadPoolExecutor;
    @Resource
    private RoomHostRecordMapper roomHostRecordMapper;

    private final HttpHeaders headers;
    private final String[] listType = new String[2];
    private String playUrl = null;

    private String getRequestUrl(){
        if(StringUtils.isBlank(playUrl)){
            SystemConfig playUrlConfig = systemConfigMapper.selectOne(Wrappers.<SystemConfig>lambdaQuery()
                    .eq(SystemConfig::getDeleted, 0)
                    .eq(SystemConfig::getConfigName, SystemConfigEnum.LIVE_PLAY_URL)
            );
            playUrl = playUrlConfig.getConfigValue();
        }
        return playUrl;
    }

    public CrawlSportLive360ServiceImpl() {
        headers = new HttpHeaders();
        headers.add("User-Agent", "Mozilla/5.0 (Linux; Android 6.0; Nexus 5 Build/MRA58N) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/89.0.4389.90 Mobile Safari/537.36");
        headers.add("Connection", "keep-alive");

        headers.setAccept(Collections.singletonList(APPLICATION_JSON));

        listType[0] = "football";
        listType[1] = "basketball";
    }

    @Override
    public void crawlScheduleMatch(String url) {
        for (String type : listType) {
            try {
                Integer categoryId = type.equals("football") ? 2 : 3;

                url = "http://" + clientID + ".api.sportlive360.com/list/" + type + "?living=1&channel=qiutan";
                ResponseEntity<String> response = restTemplate.exchange(url, HttpMethod.GET, new HttpEntity<>(null, headers), String.class);
                if (response.getStatusCode() == HttpStatus.OK) {
                    JSONObject jsonObject = JSONObject.parseObject(response.getBody());
                    List<SportLive360ListDto> live360ListDtos = JSONArray.parseArray(jsonObject.getString("data"), SportLive360ListDto.class);

                    List<Room> rooms = roomMapper.selectList(Wrappers.<Room>lambdaQuery()
                            .eq(Room::getDeleted, 0)
                            .eq(Room::getType, 1)
                            .eq(Room::getCategoryId, categoryId)
                            .likeRight(Room::getLiveUrl, getRequestUrl())
                    );

                    for (SportLive360ListDto live360ListDto : live360ListDtos) {
                        String key = RedisCommonChar.REDIS_LOCK + "live360ListDto:" + live360ListDto.getMatchId();
                        if(!redisUtilsService.existsKey(key)) {
                            redisUtilsService.cacheStringInfo(key, "1", 10);
                            crawlScheduleMatchInfo(live360ListDto, rooms);
                            redisUtilsService.removeByKey(key);
                        }
                    }

                    for (Room stop : rooms) {
                        stop.setType(0);
                        roomMapper.updateById(stop);
                    }
                }
            } catch (RuntimeException e) {
                log.error(e.getMessage());
            } finally {
                try {
                    Thread.sleep(60 * 1000L);
                } catch (InterruptedException | RuntimeException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    @Override
    public void crawlScheduleMatchByDay(String url, int day) {

    }

    @Override
    public void crawlIndexCompanyFootball(String url) {

    }

    @Override
    public void crawlIndexCompanyBasketball(String url) {

    }

    @Override
    public void crawlMatchStageScore(String url) {

    }

    @Override
    public void crawlScheduleArticle(String url) {

    }

    private void crawlScheduleMatchInfo(SportLive360ListDto live360ListDto, List<Room> rooms) {
        if (live360ListDto.getLiveStreams().size() == 0) {
            return;
        }
        String liveUrl = "";
        for (int i = 0; i < live360ListDto.getLiveStreams().size(); i++) {
            if (live360ListDto.getLiveStreams().get(i).getName().contains("中文")) {
                liveUrl = getRequestUrl() + live360ListDto.getLiveStreams().get(i).getId() + "/playlist.m3u8";
                break;
            } else if (i == live360ListDto.getLiveStreams().size() - 1) {
                liveUrl = getRequestUrl() + live360ListDto.getLiveStreams().get(i).getId() + "/playlist.m3u8";
            }
        }

        String finalLiveUrl = liveUrl;
        rooms.removeIf(deleteRoom -> deleteRoom.getLiveUrl().equals(finalLiveUrl));

        Room room = roomMapper.selectOne(Wrappers.<Room>lambdaQuery()
                .eq(Room::getDeleted, 0)
                .eq(Room::getType, 1)
                .eq(Room::getLiveUrl, liveUrl)
        );
        if (room != null) {
            return;
        }

        String roomName = live360ListDto.getLeagueName() + " " + live360ListDto.getHomeName() + "vs" + live360ListDto.getAwayName();
        Integer categoryId = live360ListDto.getCatId().equals("football") ? 2 : 3;

        Room vRoom = roomService.getVacancyRoom();
        if (vRoom == null) {
            roomService.createVacancyRoom(roomName, categoryId == 2 ? 0 : 1, liveUrl);
            List<Room> roomsList = roomMapper.selectList(Wrappers.<Room>lambdaQuery()
                    .eq(Room::getDeleted, 0)
                    .eq(Room::getRoomName, roomName)
                    .eq(Room::getLiveUrl, liveUrl)
            );
            if (roomsList.size() > 0) {
                vRoom = roomsList.get(0);
            }
        } else {
            vRoom.setRoomName(roomName);
            vRoom.setLiveUrl(liveUrl);
            vRoom.setType(1);
            vRoom.setCategoryId(categoryId);
            vRoom.setHotNum(IdGen.generatorNum(20000, 60000));
            roomMapper.updateById(vRoom);
        }

        if (vRoom == null)
            return;

        List<Match> matchs = matchMapper.selectList(Wrappers.<Match>lambdaQuery()
                .eq(Match::getDeleted, 0)
                .eq(Match::getSourceType, 1)
                .eq(Match::getMatchTime, DateUtil.format(DateUtil.getStartTime(), DateUtil.YMD_) + " " + live360ListDto.getMatchTime() + ":00")
        );

        for (Match match : matchs) {
            Team away = teamMapper.selectById(match.getAwayId());
            Team home = teamMapper.selectById(match.getHomeId());

            boolean homeBool = away.getTeamName().contains(live360ListDto.getAwayName())
                    || away.getTeamName().contains(live360ListDto.getHomeName())
                    || home.getTeamName().contains(live360ListDto.getAwayName())
                    || home.getTeamName().contains(live360ListDto.getHomeName());

            boolean awayBool = live360ListDto.getAwayName().contains(away.getTeamName())
                    || live360ListDto.getHomeName().contains(away.getTeamName())
                    || live360ListDto.getAwayName().contains(home.getTeamName())
                    || live360ListDto.getHomeName().contains(home.getTeamName());

            if (homeBool || awayBool) {
                if (StringUtils.isBlank(vRoom.getMatchId()) || !vRoom.getMatchId().equals(match.getId())) {
                    vRoom.setMatchId(match.getId());
                    roomMapper.updateById(vRoom);
                    log.info("{} -> 关联赛事 -> {}", vRoom.getHuanXinRoomId(), vRoom.getMatchId());

                    try {
                        roomHostRecordMapper.insert(RoomHostRecord.builder()
                                .receivedGold(0)
                                .roomId(vRoom.getHuanXinRoomId())
                                .matchTime(match.getMatchTime())
                                .matchId(match.getId())
                                .hostUserId(vRoom.getUserId())
                                .createTime(match.getMatchTime())
                                .endTime(DateUtil.getTmEndTime())
                                .build());
                    } catch (DuplicateKeyException e) {
                        //
                    }
                }
            }
        }
        //调整主播带上卫星前缀
        User user = userMapper.selectById(vRoom.getUserId());
        if (user != null && !user.getUserName().contains("卫星")) {
            user.setUserName("卫星" + user.getUserName());
            userMapper.updateById(user);
        }
    }

    @Override
    public void updateMatchList() {

    }

    @Override
    public void updateFootballState(String matchId, long time) {

    }

    @Override
    public void updateBasketballState(String matchId, long time) {

    }

    @Override
    public String getPlatform() {
        return CrawlPlatformEnum.SportLive360.getCode();
    }
}
